type OrderReturn = 0 | 1 | -1
export type Order<TGiven> = (a: TGiven, b: TGiven) => OrderReturn

export const make = <TGiven>(compare: Order<TGiven>) => ((a: TGiven, b: TGiven) => a === b ? 0 : compare(a, b))

export const string: Order<string> = make((a, b) => a.localeCompare(b) > 0 ? 1 : -1)

export const number: Order<number> = make((a, b) => (a < b ? -1 : 1))

export const boolean: Order<boolean> = make((a, _b) => a === false ? -1 : 1)

export const date: Order<Date> = make((a, b) => {
  const timeA = a.getTime()
  const timeB = b.getTime()
  if (timeA === timeB) {
    return 0;
  }

  return timeA < timeB ? -1 : 1
})

export const mapBy = <TGiven, TReturn>(
  order: Order<TGiven>,
  getter: (given: TReturn) => TGiven,
): Order<TReturn> => (
  make((a, b) => {
    const retrievedA = getter(a)
    const retrievedB = getter(b)
    return order(retrievedA, retrievedB)
  })
)

export const combine = <TGiven>(...orders: [Order<TGiven>, ...Array<Order<TGiven>>]) => (
  make<TGiven>((a, b) => {
    for (const order of orders) {
      const out = order(a, b)
      if (out !== 0) {
        return out
      }
    }

    return 0
  })
)

export const reverse = <TGiven>(order: Order<TGiven>): Order<TGiven> => make((a, b) => order(b, a))

